#!/usr/bin/env python3
"""
START Engagement Report Generator

This script compiles data from various CSV files to generate a comprehensive
engagement report matching the format of "START engagement report example.csv"
"""

import csv
import os
from datetime import datetime, timedelta
from collections import defaultdict
from pathlib import Path


class EngagementReportGenerator:
    def __init__(self, data_folder, output_folder=None):
        self.data_folder = Path(data_folder)
        self.output_folder = Path(output_folder) if output_folder else self.data_folder
        self.user_data = defaultdict(lambda: {
            'PID': '',
            'Onboard Date': '',
            'check-in': 0,
            'CHECK-in Past 15 Days': 0,
            '#times in app': 0,
            'Intro Welcome': 0,
            'Intro Selfcare': 0,
            'Intro Stim Use': 0,
            'PosEvents GoodThings': 0,
            'PosEve Grat Journal': 0,
            'PosEve Meditation': 0,
            'Mindful Informal Mindfulness': 0,
            'Mindful Self Compassion': 0,
            'Mindful Meditation': 0,
            'Reappraisal Reappraisal': 0,
            'Reappraisal Meditation': 0,
            'Values Intro': 0,
            'Values Values': 0,
            'Values Strengths': 0,
            'Values Goals': 0,
            'Values Meditation': 0,
            'Kindness Kindness': 0,
            'Kindness Meditation': 0,
            'Videos': 0,
            'Reminders': 0
        })

    def load_check_in_data(self):
        """Load check-in data from check-in-data.csv"""
        file_path = self.data_folder / 'check-in-data.csv'
        if not file_path.exists():
            print(f"Warning: {file_path} not found")
            return

        # Calculate date 15 days ago
        fifteen_days_ago = datetime.now() - timedelta(days=15)
        
        with open(file_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                name = row.get('Name', '').strip()
                if not name:
                    continue
                
                self.user_data[name]['check-in'] += 1
                
                # Check if submission is within past 15 days
                submitted_date_str = row.get('Submitted Date', '').strip()
                if submitted_date_str:
                    try:
                        submitted_date = datetime.strptime(submitted_date_str, '%Y-%m-%d')
                        if submitted_date >= fifteen_days_ago:
                            self.user_data[name]['CHECK-in Past 15 Days'] += 1
                    except ValueError:
                        pass

    def load_time_in_app(self):
        """Load time-in-app data from time-in-app.csv"""
        file_path = self.data_folder / 'time-in-app.csv'
        if not file_path.exists():
            print(f"Warning: {file_path} not found")
            return

        with open(file_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                name = row.get('User', '').strip()
                if not name:
                    continue
                
                self.user_data[name]['#times in app'] += 1

    def load_intro_status(self):
        """Load intro module status from intro-status.csv"""
        file_path = self.data_folder / 'intro-status.csv'
        if not file_path.exists():
            print(f"Warning: {file_path} not found")
            return

        with open(file_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                name = row.get('Name', '').strip()
                if not name:
                    continue
                
                type_val = row.get('type', '').strip()
                count = int(row.get('number_of_times_reported', 0))
                
                if type_val == 'Welcome to START':
                    self.user_data[name]['Intro Welcome'] += count
                elif type_val == 'Self Care':
                    self.user_data[name]['Intro Selfcare'] += count
                elif type_val == 'Stimulant Use & HIV':
                    self.user_data[name]['Intro Stim Use'] += count

    def load_positive_events_status(self):
        """Load positive events module status from positive-events-status.csv"""
        file_path = self.data_folder / 'positive-events-status.csv'
        if not file_path.exists():
            print(f"Warning: {file_path} not found")
            return

        with open(file_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                name = row.get('Name', '').strip()
                if not name:
                    continue
                
                type_val = row.get('type', '').strip()
                count = int(row.get('number_of_times_reported', 0))
                
                if type_val == 'Good Things':
                    self.user_data[name]['PosEvents GoodThings'] += count
                elif type_val == 'Gratitude Journaling':
                    self.user_data[name]['PosEve Grat Journal'] += count
                elif type_val == 'Good Things Meditation':
                    self.user_data[name]['PosEve Meditation'] += count

    def load_mindfulness_status(self):
        """Load mindfulness module status from mindfulness-status.csv"""
        file_path = self.data_folder / 'mindfulness-status.csv'
        if not file_path.exists():
            print(f"Warning: {file_path} not found")
            return

        with open(file_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                name = row.get('Name', '').strip()
                if not name:
                    continue
                
                type_val = row.get('type', '').strip()
                count = int(row.get('number_of_times_reported', 0))
                
                if type_val == 'Informal Mindfulness':
                    self.user_data[name]['Mindful Informal Mindfulness'] += count
                elif type_val == 'Self Compassion':
                    self.user_data[name]['Mindful Self Compassion'] += count
                elif type_val == 'Meditation Mindfulness':
                    self.user_data[name]['Mindful Meditation'] += count

    def load_reappraisal_status(self):
        """Load reappraisal module status from reappraisal-status.csv"""
        file_path = self.data_folder / 'reappraisal-status.csv'
        if not file_path.exists():
            print(f"Warning: {file_path} not found")
            return

        with open(file_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                name = row.get('Name', '').strip()
                if not name:
                    continue
                
                type_val = row.get('type', '').strip()
                count = int(row.get('number_of_times_reported', 0))
                
                if type_val == 'Reappraisal':
                    self.user_data[name]['Reappraisal Reappraisal'] += count
                elif type_val == 'Meditation Reappraisal':
                    self.user_data[name]['Reappraisal Meditation'] += count

    def load_values_status(self):
        """Load values module status from values-status.csv"""
        file_path = self.data_folder / 'values-status.csv'
        if not file_path.exists():
            print(f"Warning: {file_path} not found")
            return

        with open(file_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                name = row.get('Name', '').strip()
                if not name:
                    continue
                
                type_val = row.get('type', '').strip()
                count = int(row.get('number_of_times_reported', 0))
                
                if type_val == 'Intro Values':
                    self.user_data[name]['Values Intro'] += count
                elif type_val == 'Values':
                    self.user_data[name]['Values Values'] += count
                elif type_val == 'Strengths':
                    self.user_data[name]['Values Strengths'] += count
                elif type_val == 'Goals':
                    self.user_data[name]['Values Goals'] += count
                elif type_val == 'Meditation Values':
                    self.user_data[name]['Values Meditation'] += count

    def load_kindness_status(self):
        """Load kindness module status from kindness-status.csv"""
        file_path = self.data_folder / 'kindness-status.csv'
        if not file_path.exists():
            print(f"Warning: {file_path} not found")
            return

        with open(file_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                name = row.get('Name', '').strip()
                if not name:
                    continue
                
                type_val = row.get('type', '').strip()
                count = int(row.get('number_of_times_reported', 0))
                
                if type_val == 'Kindness':
                    self.user_data[name]['Kindness Kindness'] += count
                elif type_val == 'Meditation Kindness':
                    self.user_data[name]['Kindness Meditation'] += count

    def load_videos_watched(self):
        """Load videos watched data from videos-watched.csv"""
        file_path = self.data_folder / 'videos-watched.csv'
        if not file_path.exists():
            print(f"Warning: {file_path} not found")
            return

        with open(file_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                name = row.get('Name', '').strip()
                if not name:
                    continue
                
                count = int(row.get('number_of_times_reported', 0))
                self.user_data[name]['Videos'] = count

    def load_reminders(self):
        """Load reminders data from reminders.csv"""
        file_path = self.data_folder / 'reminders.csv'
        if not file_path.exists():
            print(f"Warning: {file_path} not found")
            return

        with open(file_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                name = row.get('User', '').strip()
                if not name:
                    continue
                
                # Count active reminders (daily or weekly)
                daily_reminder = row.get('Daily Reminder', '').strip()
                weekly_reminder = row.get('Weekly Reminder', '').strip()
                
                # Check if either daily or weekly reminder is active (marked with ✔)
                if daily_reminder == '✔' or weekly_reminder == '✔':
                    self.user_data[name]['Reminders'] += 1

    def load_all_data(self):
        """Load all CSV files and compile the data"""
        print("Loading check-in data...")
        self.load_check_in_data()
        
        print("Loading time in app data...")
        self.load_time_in_app()
        
        print("Loading intro status...")
        self.load_intro_status()
        
        print("Loading positive events status...")
        self.load_positive_events_status()
        
        print("Loading mindfulness status...")
        self.load_mindfulness_status()
        
        print("Loading reappraisal status...")
        self.load_reappraisal_status()
        
        print("Loading values status...")
        self.load_values_status()
        
        print("Loading kindness status...")
        self.load_kindness_status()
        
        print("Loading videos watched...")
        self.load_videos_watched()
        
        print("Loading reminders...")
        self.load_reminders()

    def generate_report(self, output_file='engagement_report.csv'):
        """Generate the engagement report CSV file"""
        output_path = self.output_folder / output_file
        
        # Define column headers
        headers = [
            'Name', 'PID', 'Onboard Date', 'check-in', 'CHECK-in Past 15 Days', '#times in app',
            'Intro Welcome', 'Intro Selfcare', 'Intro Stim Use',
            'PosEvents GoodThings', 'PosEve Grat Journal', 'PosEve Meditation',
            'Mindful Informal Mindfulness', 'Mindful Self Compassion', 'Mindful Meditation',
            'Reappraisal Reappraisal', 'Reappraisal Meditation',
            'Values Intro', 'Values Values', 'Values Strengths', 'Values Goals', 'Values Meditation',
            'Kindness Kindness', 'Kindness Meditation',
            'Videos', 'Reminders'
        ]
        
        # Sort users alphabetically
        sorted_users = sorted(self.user_data.keys())
        
        with open(output_path, 'w', newline='', encoding='utf-8') as f:
            writer = csv.DictWriter(f, fieldnames=headers)
            
            # Write header row
            writer.writeheader()
            
            # Write comment row (matching example format)
            comment_row = {
                'Name': '',
                'PID': '# FROM STUDY TEAM',
                'Onboard Date': 'YYYYMMDD',
                'check-in': '4',
                'CHECK-in Past 15 Days': '',
                '#times in app': '3',
                'Intro Welcome': '1',
                'Intro Selfcare': '2',
                'Intro Stim Use': '1',
                'PosEvents GoodThings': '1',
                'PosEve Grat Journal': '0',
                'PosEve Meditation': '0',
                'Mindful Informal Mindfulness': '0',
                'Mindful Self Compassion': '0',
                'Mindful Meditation': '0',
                'Reappraisal Reappraisal': '0',
                'Reappraisal Meditation': '0',
                'Values Intro': '0',
                'Values Values': '0',
                'Values Strengths': '0',
                'Values Goals': '0',
                'Values Meditation': '0',
                'Kindness Kindness': '0',
                'Kindness Meditation': '0',
                'Videos': '12',
                'Reminders': '0'
            }
            writer.writerow(comment_row)
            
            # Write data rows
            for user in sorted_users:
                data = self.user_data[user]
                row = {
                    'Name': user,
                    'PID': '',  # Leave blank as in example
                    'Onboard Date': data['Onboard Date'],
                    'check-in': data['check-in'],
                    'CHECK-in Past 15 Days': '',  # Leave blank as in example
                    '#times in app': data['#times in app'],
                    'Intro Welcome': data['Intro Welcome'],
                    'Intro Selfcare': data['Intro Selfcare'],
                    'Intro Stim Use': data['Intro Stim Use'],
                    'PosEvents GoodThings': data['PosEvents GoodThings'],
                    'PosEve Grat Journal': data['PosEve Grat Journal'],
                    'PosEve Meditation': data['PosEve Meditation'],
                    'Mindful Informal Mindfulness': data['Mindful Informal Mindfulness'],
                    'Mindful Self Compassion': data['Mindful Self Compassion'],
                    'Mindful Meditation': data['Mindful Meditation'],
                    'Reappraisal Reappraisal': data['Reappraisal Reappraisal'],
                    'Reappraisal Meditation': data['Reappraisal Meditation'],
                    'Values Intro': data['Values Intro'],
                    'Values Values': data['Values Values'],
                    'Values Strengths': data['Values Strengths'],
                    'Values Goals': data['Values Goals'],
                    'Values Meditation': data['Values Meditation'],
                    'Kindness Kindness': data['Kindness Kindness'],
                    'Kindness Meditation': data['Kindness Meditation'],
                    'Videos': data['Videos'],
                    'Reminders': data['Reminders']
                }
                writer.writerow(row)
        
        print(f"\nReport generated successfully: {output_path}")
        print(f"Total users: {len(sorted_users)}")
        return output_path


def main():
    """Main function to run the report generator"""
    # Get the script's directory
    script_dir = Path(__file__).parent
    data_dir = script_dir / 'data'
    
    print("=" * 60)
    print("START Engagement Report Generator")
    print("=" * 60)
    print()
    
    # Create generator instance
    generator = EngagementReportGenerator(data_dir, output_folder=script_dir)
    
    # Load all data
    generator.load_all_data()
    
    # Generate report
    output_file = f'engagement_report_{datetime.now().strftime("%Y%m%d_%H%M%S")}.csv'
    generator.generate_report(output_file)
    
    print()
    print("=" * 60)
    print("Report generation complete!")
    print("=" * 60)


if __name__ == '__main__':
    main()
